import * as echart from 'echarts'
import { onUnmounted, shallowReactive } from 'vue'
export const getEchartImage = (option: {
  width: string | number
  height: string | number
  chartOption: any
  theme?: string
  backgroundColor?: string
}) => {
  const { chartOption, height, width, backgroundColor = '#fff', theme } = option
  const dom = document.createElement('div')
  dom.style.height = height + ''
  dom.style.width = width + ''
  const instance = echart.init(dom, theme)
  instance.setOption(chartOption)
  instance.resize() // 调整图表大小
  return instance.getDataURL({ backgroundColor })
}

export const useChartHighlight = () => {
  const hignLightInfo = shallowReactive({
    /** 图表实例 */
    chartInstance: null as echarts.ECharts | null,
    /** 数据的个数 */
    len: 0,
    inited: false,
    /** 动画标识 */
    identifier: -1,
    /** 记录时间 */
    time: 0,
    /** 当前自动到第几个 */
    currentIndex: 0,
    /** 第几个series 默认0 */
    seriesIndex: 0 as number | number[]
  })

  const requestAnimationFunction = (timeStamp: number) => {
    const { currentIndex, len, time, chartInstance, seriesIndex } = hignLightInfo
    if (timeStamp > time + 1000) {
      const lastIndex = currentIndex === 0 ? len - 1 : currentIndex - 1
      // 如果要取消高亮系列：
      chartInstance?.dispatchAction({ type: 'downplay', seriesIndex, dataIndex: lastIndex })
      chartInstance?.dispatchAction({
        type: 'showTip', // 根据 tooltip 的配置项显示提示框。
        // 系列的 index，在 tooltip 的 trigger 为 axis 的时候可选。
        seriesIndex: typeof seriesIndex === 'number' ? seriesIndex || 0 : 0,
        dataIndex: currentIndex
      })
      chartInstance?.dispatchAction({ type: 'highlight', seriesIndex, dataIndex: currentIndex })
      hignLightInfo.currentIndex = currentIndex + 1
      if (currentIndex === len) {
        hignLightInfo.currentIndex = 0
      }
      hignLightInfo.time = timeStamp
    }
    hignLightInfo.identifier = window.requestAnimationFrame(requestAnimationFunction)
  }
  const cancelAnimation = () => {
    const { currentIndex, len, seriesIndex } = hignLightInfo
    const lastIndex = currentIndex === 0 ? len - 1 : currentIndex - 1
    hignLightInfo.chartInstance?.dispatchAction({
      type: 'downplay',
      seriesIndex,
      dataIndex: lastIndex
    })
    window.cancelAnimationFrame(hignLightInfo.identifier)
  }
  const startAnimation = () => {
    const { len, identifier } = hignLightInfo
    window.cancelAnimationFrame(identifier)
    len > 0 && (hignLightInfo.identifier = window.requestAnimationFrame(requestAnimationFunction))
  }
  onUnmounted(() => {
    cancelAnimation()

    const { chartInstance } = hignLightInfo
    chartInstance?.getDom()?.removeEventListener('mouseenter', cancelAnimation)
    chartInstance?.getDom()?.removeEventListener('mouseleave', startAnimation)
  })

  function chartHignlight(option: {
    /** 第几个series 默认0 */
    seriesIndex?: number | number[]
    chartInstance: echarts.ECharts
    /** 数据的个数 */
    len: number
    /** 不立即执行 */
    notImmediate?: boolean
  }) {
    if (hignLightInfo.inited && option.chartInstance === hignLightInfo.chartInstance)
      return console.warn('不能重复调用')
    if (!option.chartInstance) return console.warn('请传入chart实例')
    if (!option.len || typeof option.len !== 'number') return console.warn('请传入正确的数据个数')
    hignLightInfo.chartInstance = option.chartInstance
    hignLightInfo.len = option.len || 0
    hignLightInfo.seriesIndex =
      typeof option.seriesIndex === 'number' ? option.seriesIndex || 0 : option.seriesIndex || [0]
    hignLightInfo.chartInstance.getDom().addEventListener('mouseenter', cancelAnimation)
    hignLightInfo.chartInstance.getDom().addEventListener('mouseleave', startAnimation)
    hignLightInfo.inited = true
    if (option.notImmediate) return () => startAnimation()
    startAnimation()
  }

  return { chartHignlight }
}
