import type { Edge, Graph } from '@antv/x6'
import { Stencil } from '@antv/x6-plugin-stencil'
import {
  addPortsForDragNode,
  createImageNode as originalCreateImageNode,
  type GraphOptions
} from '../'
import type { ImageNodeOptionI } from '../constants'
import { createBorderNode, createTextNode } from '../create'
export const nodePortOptions = {
  ports: {
    groups: {
      left: {
        position: 'left',
        attrs: {
          circle: {
            magnet: true,
            stroke: '#8f8f8f',
            r: 4
          }
        },
        label: {
          position: 'left'
        }
      },

      top: {
        position: 'top',
        attrs: {
          circle: {
            magnet: true,
            stroke: '#8f8f8f',
            r: 4
          }
        },
        label: {
          position: 'top'
        }
      },
      right: {
        position: 'right',
        attrs: {
          circle: {
            magnet: true,
            stroke: '#8f8f8f',
            r: 4
          }
        },
        label: {
          position: 'right'
        }
      },
      bottom: {
        position: 'bottom',
        attrs: {
          circle: {
            magnet: true,
            stroke: '#8f8f8f',
            r: 4
          }
        },
        label: {
          position: 'top'
        }
      }
    }
  }
}

export const createImageNode = (options: ImageNodeOptionI) =>
  originalCreateImageNode({ ...nodePortOptions, isHolder: true, ...options })

/** 普通节点 */
const generalTypes = ['node', 'container']
const needIpTypes = ['network', 'route']
export const graphOptions = {
  autoResize: true,
  background: {
    // color: '#F2F7FA'
  },
  mousewheel: {
    enabled: true,
    modifiers: ['ctrl']
  },

  // virtual: true,
  grid: {
    visible: true,
    type: 'doubleMesh',
    args: [
      {
        color: '#eee', // 主网格线颜色
        thickness: 1 // 主网格线宽度
      },
      {
        color: '#ddd', // 次网格线颜色
        thickness: 1, // 次网格线宽度
        factor: 4 // 主次网格线间隔
      }
    ]
  },
  // 连线
  connecting: {
    // connectionPoint: {
    //   name: 'bbox',
    //   args: {
    //     sticky: true
    //   }
    // },

    // 吸附
    snap: true,
    allowBlank: false,
    allowLoop: false,
    allowEdge: false,
    allowNode: false,
    allowPort({ sourceCell, targetCell }) {
      const sourceType = sourceCell?.getProp('type')
      const targetType = targetCell?.getProp('type')
      // 节点和节点之间不能互相连接
      if (generalTypes.includes(sourceType) && generalTypes.includes(targetType)) {
        return false
      }
      // 当是流量告警设备时，不能和普通节点连接
      if (
        (sourceType === 'device' && targetType === 'device') ||
        (sourceType === 'device' && generalTypes.includes(targetType)) ||
        (targetType === 'device' && generalTypes.includes(sourceType))
      ) {
        return false
      }

      // 当是物理网络时，只能和节点或者容器连接
      if (
        (sourceType === 'physics' && targetType === 'physics') ||
        (sourceType === 'physics' && !generalTypes.includes(targetType)) ||
        (targetType === 'physics' && !generalTypes.includes(sourceType))
      ) {
        return false
      }

      return true
    },
    highlight: true
  }
} as Partial<GraphOptions.Manual>

export function registerStencil(graph: Graph) {
  const stencil = new Stencil({
    title: '元素',
    target: graph,
    stencilGraphWidth: 200,
    stencilGraphHeight: 0,
    stencilGraphPadding: 20,
    groups: [
      // { title: '默认', name: 'default' },
      { title: '文本', name: 'text' },
      { title: '节点', name: 'node' }
    ],
    getDropNode(_) {
      const node = _.clone()
      node.prop('isHolder', null)
      if (node.shape === 'border-node') {
        node.prop('size/width', 200)
        node.prop('size/height', 160)
      }
      node.getProp('isCanLink') && addPortsForDragNode(node)
      const type = node.getProp('type')
      const label = node.getProp('label')

      // 判断同一种的话后面个数滚动
      const count = graph
        .getNodes()
        .filter(
          n =>
            n.getProp('type') === type && n.getProp('label').split('_')[0] === node.getProp('label')
        ).length
      node.setProp('label', `${label}_${count + 1}`)

      return node
    }
  })
  // 需要一个容纳 stencil 的 Dom 容器 stencilContainer
  const stencilContainer = document.querySelector('.x6-stencil-container') as HTMLDivElement

  stencilContainer.appendChild(stencil.container)
  const text = graph.createNode(createTextNode({ label: '文本', isHolder: true }))
  const border = graph.createNode(
    createBorderNode({ label: '边框', width: 48, height: 48, isHolder: true })
  )
  stencil.load([text, border], 'text')
  stencil.load(
    [
      graph.createNode(createImageNode({ label: '节点', type: 'node' })),
      graph.createNode(createImageNode({ label: '设备', type: 'device' })),
      graph.createNode(createImageNode({ label: '交换机', type: 'network' })),
      graph.createNode(createImageNode({ label: '路由器', type: 'route' }))
    ],
    'node'
  )
  return stencil
}

/**
 * 检查和设置连线的data
 * @param edge
 */
export function checkEdgeData(edge: Edge<Edge.Properties>) {
  const { line: { stroke = '#333333', strokeWidth = 2 } = {} } = edge.getProp('attrs') || {}

  const data = edge.getData() // 如果不在新增时设置，此项是undefined，考虑撤销功能或者导入功能
  if (!data) {
    edge.setData({
      stroke,
      strokeWidth
    } as DragView.EdgeData)
  } else {
    edge.setData(data)
  }
}

export function registerEvents(graph: Graph) {
  // 撤销重做时重新添加或初始添加时需要把showPorts的逻辑加上
  graph.on('cell:added', ({ cell }) => {
    if (cell.isEdge()) {
      cell.prop('attrs/line/targetMarker', null)
      checkEdgeData(cell)
    }
  })

  // 连接时添加ip占位字段
  graph.on('edge:connected', ({ edge }) => {
    const targetNode = edge.getTargetNode()
    const targetType = targetNode?.getProp('type')
    const sourceNode = edge.getSourceNode()
    const sourceType = sourceNode?.getProp('type')
    // 如果两端的某一端是交换机或路由，另一端是节点或容器
    if (needIpTypes.includes(targetType) && generalTypes.includes(sourceType)) {
      const data = sourceNode?.getData() || {}
      const ipInfos = data.ipInfos ? [...data.ipInfos] : []
      ipInfos.push({
        // 目标交换机或路由的id
        target: targetNode?.id,
        label: targetNode?.getProp('label'),
        ip: '',
        autoIp: true
      })
      sourceNode?.setData({ ...data, ipInfos })
    }

    if (needIpTypes.includes(sourceType) && generalTypes.includes(targetType)) {
      const data = targetNode?.getData() || {}
      const ipInfos = data.ipInfos ? [...data.ipInfos] : []
      ipInfos.push({
        target: sourceNode?.id,
        label: sourceNode?.getProp('label'),
        ip: '',
        autoIp: true
      })
      targetNode?.setData({ ...data, ipInfos })
    }
  })
  // 删除时移除ip占位字段
  graph.on('edge:removed', ({ cell }) => {
    if (cell.isEdge()) {
      // @ts-ignore
      const targetNodeId = cell.target.cell || ''
      const targetNode = graph.getCellById(targetNodeId)
      const targetType = targetNode?.getProp('type')
      // @ts-ignore
      const sourceNodeId = cell.source.cell || ''
      const sourceNode = graph.getCellById(sourceNodeId)
      const sourceType = sourceNode?.getProp('type')
      // 如果两端的某一端是交换机或路由，另一端是节点或容器
      if (needIpTypes.includes(targetType) && generalTypes.includes(sourceType)) {
        const data = sourceNode?.getData() || {}
        const ipInfos = [...data.ipInfos]
        const index = ipInfos.findIndex(item => item.target === targetNode?.id)
        if (index != -1) {
          ipInfos.splice(index, 1)
        }
        sourceNode?.replaceData({ ...data, ipInfos })
      }

      if (needIpTypes.includes(sourceType) && generalTypes.includes(targetType)) {
        const data = targetNode?.getData() || {}
        const ipInfos = [...data.ipInfos]
        const index = ipInfos.findIndex(item => item.target === sourceNode?.id)
        if (index != -1) {
          ipInfos.splice(index, 1)
        }
        targetNode?.replaceData({ ...data, ipInfos })
      }
    }
  })

  graph.on('edge:change:data', ({ edge, current }) => {
    const { stroke, strokeWidth } = current as DragView.EdgeData
    edge.prop('attrs/line', { stroke, strokeWidth })
  })
}
