import { Graph } from '@antv/x6'
import { Clipboard } from '@antv/x6-plugin-clipboard'
import { Export } from '@antv/x6-plugin-export'
import { History } from '@antv/x6-plugin-history'
import { Keyboard } from '@antv/x6-plugin-keyboard'
import { MiniMap } from '@antv/x6-plugin-minimap'
import { Scroller } from '@antv/x6-plugin-scroller'
import { Selection } from '@antv/x6-plugin-selection'
import { Snapline } from '@antv/x6-plugin-snapline'
import { Stencil } from '@antv/x6-plugin-stencil'
import { Transform } from '@antv/x6-plugin-transform'
import { registerPortFunctionForDragNode } from './tools'
export function registerGraphPlugin(graph: Graph) {
  // 判断节点是否禁用移动
  graph.options.interacting = ({ cell }) => {
    if (cell.getProp('disableMove')) {
      return { nodeMovable: false }
    }
    return true
  }

  // 对齐辅助线
  graph.use(new Snapline({ enabled: true, tolerance: 1, resizing: true }))
  // 复制粘贴
  graph.use(new Clipboard({ enabled: true }))

  // 变换
  graph.use(
    new Transform({
      resizing: {
        enabled: node => !node.getProp('disableTransform'),
        minHeight: 20,
        minWidth: 40,
        // 如果是图片节点，按比例缩放
        preserveAspectRatio: node => node.shape === 'image-node'
      },
      rotating: { enabled: false, grid: 10 }
    })
  )
  // 框选
  graph.use(
    new Selection({
      className: 'select',
      enabled: true,
      multiple: true,
      rubberband: true,
      movable: true,
      showNodeSelectionBox: true,
      showEdgeSelectionBox: true,
      filter(cell) {
        // 编辑时不能框选也不能变换
        if (cell.getProp('disableSelect')) return false
        return true
      }
    })
  )

  graph.whenHistoryChangeIgnoreKeys = new Set([
    'ports',
    'disableTransform',
    'disableSelect',
    'disableMove',
    'disableProts'
  ])
  // 撤销/回退
  graph.use(
    new History({
      enabled: true,
      stackSize: 1000,
      beforeAddCommand(event, args) {
        if (event === 'cell:change:*') {
          // @ts-ignore
          const targetKey = args?.key as string
          if (graph.whenHistoryChangeIgnoreKeys.has(targetKey)) {
            return false
          }
        }
      }
    })
  )
  // 滚动画布
  graph.use(
    new Scroller({
      enabled: true,
      pannable: true,
      modifiers: 'alt&shift',
      pageWidth: 300,
      pageHeight: 200,
      padding: 200
    })
  )
  // 小地图
  graph.use(
    new MiniMap({
      container: document.querySelector('.x6-minimap-container') as HTMLDivElement
    })
  )
  registerKeyboard(graph)
}

export function registerStencil(graph: Graph) {
  const stencil = new Stencil({
    title: '元素',
    target: graph,
    stencilGraphWidth: 200,
    stencilGraphHeight: 0,
    stencilGraphPadding: 20,
    groups: [{ name: '默认' }]
  })
  // 需要一个容纳 stencil 的 Dom 容器 stencilContainer
  const stencilContainer = document.querySelector('.x6-stencil-container') as HTMLDivElement
  stencilContainer.appendChild(stencil.container)
  const text = graph.createNode({ shape: 'rect', width: 40, height: 40 })
  stencil.load([text], '默认')
}

export function registerKeyboard(graph: Graph) {
  graph.use(
    new Keyboard({
      enabled: true
    })
  )
  graph.bindKey('ctrl+c', () => {
    const cells = graph.getSelectedCells()
    if (cells.length) {
      graph.copy(cells)
    }
    return false
  })

  graph.bindKey('ctrl+v', () => {
    if (!graph.isClipboardEmpty()) {
      const cells = graph.paste({ offset: 32 })
      cells.forEach(cell => cell.isNode() && registerPortFunctionForDragNode(cell))
      graph.cleanSelection()
      graph.select(cells)
    }
    return false
  })

  graph.bindKey('ctrl+z', () => {
    graph?.undo()
    return false
  })

  graph.bindKey('ctrl+y', () => {
    graph?.redo()
    return false
  })

  graph.bindKey('ctrl+a', e => {
    e.preventDefault()
    e.stopImmediatePropagation()
    graph.select(graph.getCells())
  })

  graph.bindKey('up', () => {
    const ignoreTagNames = ['INPUT', 'TEXTAREA']
    if (ignoreTagNames.includes(document.activeElement?.tagName || '')) {
      return true
    }

    const cells = graph.getSelectedCells()
    cells.forEach(cell => {
      if (cell.isNode()) {
        const { x, y } = cell.position()
        cell.position(x, y - 1)
      }
    })
    return false
  })
  graph.bindKey('right', () => {
    const ignoreTagNames = ['INPUT', 'TEXTAREA']
    if (ignoreTagNames.includes(document.activeElement?.tagName || '')) {
      return true
    }
    const cells = graph.getSelectedCells()
    cells.forEach(cell => {
      if (cell.isNode()) {
        const { x, y } = cell.position()
        cell.position(x + 1, y)
      }
    })
    return false
  })
  graph.bindKey('down', () => {
    const ignoreTagNames = ['INPUT', 'TEXTAREA']
    if (ignoreTagNames.includes(document.activeElement?.tagName || '')) {
      return true
    }
    const cells = graph.getSelectedCells()
    cells.forEach(cell => {
      if (cell.isNode()) {
        const { x, y } = cell.position()
        cell.position(x, y + 1)
      }
    })
    return false
  })
  graph.bindKey('left', () => {
    const ignoreTagNames = ['INPUT', 'TEXTAREA']
    if (ignoreTagNames.includes(document.activeElement?.tagName || '')) {
      return true
    }
    const cells = graph.getSelectedCells()
    cells.forEach(cell => {
      if (cell.isNode()) {
        const { x, y } = cell.position()
        cell.position(x - 1, y)
      }
    })
    return false
  })

  function _deleteNode() {
    if (!graph.isSelectionEnabled()) return
    if (graph.isSelectionEmpty()) return
    const cells = graph.getSelectedCells()
    cells.forEach(cell => cell.remove())
    graph.cleanSelection()
  }
  graph.bindKey('delete', _deleteNode)
  graph.bindKey('backspace', _deleteNode)
  // graph.bindKey('ctrl+s', e => {
  //   e.preventDefault()
  //   e.stopImmediatePropagation()
  //   graph.trigger('view:save')
  // })
}

export function registerExport(graph: Graph) {
  graph.use(new Export())
}
