import { EdgeView, ToolsView } from '@antv/x6'
import { render as VueRender, h } from 'vue'
import ContextMenu from './ContextMenu.vue'

export interface ContextMenuToolOptions extends ToolsView.ToolItem.Options {}

export class ContextMenuTool extends ToolsView.ToolItem<EdgeView, ContextMenuToolOptions> {
  private knob!: HTMLDivElement
  private timer!: number

  render() {
    if (!this.knob) {
      this.knob = ToolsView.createElement('div', false) as HTMLDivElement
      this.knob.style.position = 'absolute'
      this.container.appendChild(this.knob)
    }
    return this
  }

  private toggleContextMenu(visible: boolean) {
    VueRender(h('div'), this.knob)
    // ReactDom.unmountComponentAtNode(this.knob)
    document.removeEventListener('mousedown', this.onMouseDown)

    if (visible) {
      // 每次需要重新生成并挂载到knob这个dom中去
      VueRender(h(ContextMenu, { ctx: this }), this.knob)
      document.addEventListener('mousedown', this.onMouseDown)
    }
  }

  private updatePosition(e?: MouseEvent) {
    const style = this.knob.style
    if (e) {
      const pos = this.graph.clientToGraph(e.clientX, e.clientY)
      style.left = `${pos.x}px`
      style.top = `${pos.y}px`
    } else {
      style.left = '-1000px'
      style.top = '-1000px'
    }
  }

  private onMouseDown = () => {
    this.timer = window.setTimeout(() => {
      this.updatePosition()
      this.toggleContextMenu(false)
    }, 200)
  }

  private onContextMenu({ e }: { e: MouseEvent }) {
    // 如果是自定义的只读，不显示
    if (this.graph.readonly) return

    if (this.timer) {
      clearTimeout(this.timer)
      this.timer = 0
    }
    this.updatePosition(e)
    this.toggleContextMenu(true)
  }

  delegateEvents() {
    this.cellView.on('cell:contextmenu', this.onContextMenu, this)
    return super.delegateEvents()
  }

  protected onRemove() {
    this.cellView.off('cell:contextmenu', this.onContextMenu, this)
  }
}
